using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace ProTerminal.Api.Example;

internal sealed class RouterClient
{
    public Task<JObject> Request(string channel, dynamic request)
    {
        var req = new RoutingRequest {
            Channel = channel,
            Command = "request",
            Id      = Guid.NewGuid().ToString(),
            Payload = JObject.FromObject(request)
                             .ToString(Formatting.None),
        };

        var tcs = new TaskCompletionSource<JObject>();

        _connection.WhenReceived += onReceived;

        _connection.Send(JObject.FromObject(req));

        return tcs.Task;

        void onReceived(object? _, JObject e)
        {
            var res = e.ToObject<RoutingRequest>();

            if (res?.Payload is null)
            {
                return;
            }

            if (res.Command != "response")
            {
                return;
            }

            if (res.Channel != req.Channel)
            {
                return;
            }

            if (res.Id != req.Id)
            {
                return;
            }

            tcs.TrySetResult(JObject.Parse(res.Payload));
        }
    }

    public void Listen(string channel, Action<JObject> handler)
    {
        var req = new RoutingRequest {
            Channel = channel,
            Command = "listen",
            Id = Guid.NewGuid().ToString(),
        };

        _connection.WhenReceived += onReceived;

        _connection.Send(JObject.FromObject(req));

        return;

        void onReceived(object? _, JObject e)
        {
            var res = e.ToObject<RoutingRequest>();

            if (res?.Payload is null)
            {
                return;
            }

            if (res.Command != "broadcast")
            {
                return;
            }

            if (res.Channel != req.Channel)
            {
                return;
            }

            var obj = JObject.Parse(res.Payload);

            handler(obj);
        }
    }


    public void UnListen(string channel)
    {
        var req = new RoutingRequest {
            Channel = channel,
            Command = "unlisten",
            Id      = Guid.NewGuid().ToString(),
        };

        _connection.Send(JObject.FromObject(req));

    }

    public sealed class BroadcastArgs : EventArgs
    {
        public required string  Channel { get; init; }
        public required JObject Payload { get; init; }
    }

    public event EventHandler<BroadcastArgs>? WhenBroadcasted;

    public RouterClient()
    {
        _connection              =  new();
        _connection.WhenReceived += OnReceived;
    }

    private void OnReceived(object? sender, JObject e)
    {
        var res = e.ToObject<RoutingRequest>();

        if (res?.Payload is null)
        {
            return;
        }

        if (res.Command != "broadcast")
        {
            return;
        }

        try
        {
            var payload = JObject.Parse(res.Payload);

            WhenBroadcasted?.Invoke(this,
                                    new() {
                                        Channel = res.Channel,
                                        Payload = payload,
                                    });
        }
        catch
        {
            //
        }
    }

    private readonly WebSocketConnection _connection;
}
